/***************************************************************************/
/*                                                                         */
/*  idrv.c                                                                 */
/*                                                                         */
/*  Copyright (C) 2012 Monotype Imaging Inc. All rights reserved.          */
/*                                                                         */
/***************************************************************************/

#include <internal/internal.h>
#include FT_INTERNAL_DEBUG_H
#include FT_TRUETYPE_IDS_H
#include FT_INTERNAL_OBJECTS_H
#include FT_INTERNAL_SFNT_H
#include FT_OUTLINE_H
#include FT_INTERNAL_TRUETYPE_TYPES_H
#include FT_INTERNAL_VALIDATE_H
#include FT_TRUETYPE_TAGS_H
#include FT_SERVICE_TT_CMAP_H


#ifdef FT_UNUSED
#undef FT_UNUSED
#endif
#define FT_UNUSED(arg) (void)arg /* to suppress Lint errors */

#include "idrv_fio.h"
#include "fs_api.h"
#include "fs_ltt.h"
#include "idrv.h"
#include "idrverrs.h"

#ifdef TXTC_BUILDS
#define LOG_TAG "itype"
#include <utils/Log.h>      /* write to logcat */
#define ANDROID_LOG_DEBUG 3 /* write to logcat */
#define ANDROID_LOG_INFO  4 /* write to logcat */

void print_config_info( FS_BOOLEAN RTGAH_FLAG,
                        FS_ULONG heap_size,
                        FS_ULONG allocated ); /* write to logcat */
#ifdef FS_HINTS
void print_RTGAH_info( FS_ULONG error, FS_BYTE hints );
#endif
#endif

#ifdef FT_CONFIG_OPTION_PIC
#error "FT_CONFIG_OPTION_PIC is currently not supported"
#endif

/*
 * defining this macro will display iType and Freetype outline information
 */
#undef DUMP_OUTLINES
#ifdef DUMP_OUTLINES
void itype_drv_dump_itype_outline( FS_OUTLINE *outl, char *msg );
void itype_drv_dump_freetype_outline( FT_Outline *outl, char *msg );
#endif


/*lint -e717*/ /* for the do..while in FT_FREE */


/*************************************************************************/
/*                                                                       */
/*                     CMAP related functions                            */
/*                                                                       */
/*************************************************************************/

typedef struct  ITYPE_DRV_CMapRec_
{
    FT_CMapRec        cmap;
    FT_UInt           num_encodings;
    ITYPE_DRV_encoding_el*  encodings;

} ITYPE_DRV_CMapRec;

/* see TT_CMap_ValidateFunc */
typedef FT_Error
  (*ITC_CMap_ValidateFunc)( FT_Byte*      data,
                            FT_Validator  valid );

/* see TT_CMap_ClassRec */
typedef struct  ITC_CMap_ClassRec_
{
    FT_ULong               size;
    FT_CMap_InitFunc       init;
    FT_CMap_DoneFunc       done;
    FT_CMap_CharIndexFunc  char_index;
    FT_CMap_CharNextFunc   char_next;

    /* Subsequent entries are special ones for format 14 -- the variant */
    /* selector subtable which behaves like no other                    */

    FT_CMap_CharVarIndexFunc      char_var_index;
    FT_CMap_CharVarIsDefaultFunc  char_var_default;
    FT_CMap_VariantListFunc       variant_list;
    FT_CMap_CharVariantListFunc   charvariant_list;
    FT_CMap_VariantCharListFunc   variantchar_list;

    /* Subsequent entries match TT_CMap_ClassRec */
    FT_UInt               format;
    ITC_CMap_ValidateFunc validate;
    TT_CMap_Info_GetFunc  get_cmap_info;

} ITC_CMap_ClassRec;

/*************************************************************************/
/*                                                                       */
/*                     itype_drv_get_cmap_info()                         */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
  itype_drv_get_cmap_info( FT_CharMap    charmap,
                           TT_CMapInfo  *cmap_info )
{
    FT_Error  error = ITYPE_DRV_Err_Ok;

    cmap_info->language = 0;
    cmap_info->format   = 0;

    if( charmap->encoding_id == 1 )
        cmap_info->format = 4;
    else if( charmap->encoding_id == 10 )
        cmap_info->format = 12;

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                     itype_drv_cmap_validate()                         */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_cmap_validate( FT_Byte*      data,
                         FT_Validator  valid )
{
    FT_Error  error = ITYPE_DRV_Err_Ok;
    FT_UNUSED(data);
    FT_UNUSED(valid);
    return error;
}

/*************************************************************************/
/*                                                                       */
/*                       itype_drv_cmap_init()                           */
/*                                                                       */
/*************************************************************************/
FT_CALLBACK_DEF( FT_Error )
itype_drv_cmap_init( FT_CMap itype_drv_cmap, FT_Pointer init_data )
{
    FT_Error error = ITYPE_DRV_Err_Ok;

    FT_UNUSED(itype_drv_cmap);
    FT_UNUSED(init_data);

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                        itype_drv_cmap_done()                          */
/*                                                                       */
/*************************************************************************/
FT_CALLBACK_DEF( void )
itype_drv_cmap_done( FT_CMap itype_drv_cmap )
{
    FT_UNUSED(itype_drv_cmap);
}

/*************************************************************************/
/*                                                                       */
/*                       itype_drv_cmap_char_index()                     */
/*                                                                       */
/*************************************************************************/
FT_CALLBACK_DEF( FT_UInt )
itype_drv_cmap_char_index( FT_CMap itype_drv_cmap, FT_UInt32 charcode )
{
    FT_UInt index = 0;

    if( VALID_MTI_EXTENSION(itype_drv_cmap->charmap.face->extensions) )
    {
        FS_STATE *FS_state_ptr;
        FS_LONG itype_err;
        FS_USHORT platform = 0;
        FS_USHORT encoding = 0;
        MTI_Info ext = (MTI_Info)itype_drv_cmap->charmap.face->extensions;

        FS_state_ptr = ext->face_client;

        if( FS_state_ptr != NULL )
        {
            switch( itype_drv_cmap->charmap.encoding )
            {
                case FT_ENCODING_MS_SYMBOL:
                    platform = 3;
                    encoding = 0;
                    break;

                case FT_ENCODING_UNICODE:
                    platform = 3;
                    encoding = 1;
                    break;

                case FT_ENCODING_SJIS:
                    platform = 3;
                    encoding = 2;
                    break;

                case FT_ENCODING_GB2312:
                    platform = 3;
                    encoding = 3;
                    break;

                case FT_ENCODING_BIG5:
                    platform = 3;
                    encoding = 4;
                    break;

               case FT_ENCODING_WANSUNG:
                    platform = 3;
                    encoding = 5;
                    break;

               case FT_ENCODING_JOHAB:
                   platform = 3;
                   encoding = 6;
                   break;

               case FT_ENCODING_NONE:
                   default:
                   /* will generate an error */
                   platform = 9999;
                   encoding = 9999;
                   break;
            }

            itype_err = FS_set_cmap( FS_state_ptr, platform, encoding );

            if( (itype_err == SUCCESS) || (itype_err == ERR_CMAP_UNSUPPORTED) )
                index = FS_map_char( FS_state_ptr, charcode );
        }
    }

    return index;
}

/*************************************************************************/
/*                                                                       */
/*                      itype_drv_cmap_char_next()                       */
/*                                                                       */
/*************************************************************************/
FT_CALLBACK_DEF( FT_UInt )
itype_drv_cmap_char_next( FT_CMap     itype_drv_cmap,
                          FT_UInt32  *acharcode )
{
    FS_STATE *FS_state_ptr = NULL;
    FT_UInt   gindex = 0;
    FT_UInt32 unicode = *acharcode;

    if( VALID_MTI_EXTENSION(itype_drv_cmap->charmap.face->extensions) )
    {
        MTI_Info ext = (MTI_Info)itype_drv_cmap->charmap.face->extensions;
        FS_state_ptr = ext->face_client;

        if( FS_state_ptr != NULL )
        {
            for( unicode += 1; unicode < 0xFFFF; unicode++ )
            {
                gindex = FS_map_char( FS_state_ptr, unicode );
                if( gindex )
                {
                    *acharcode = unicode;
                    return gindex;
                }
            }
        }
    }

    return gindex;
}

FT_CALLBACK_TABLE_DEF
const ITC_CMap_ClassRec  itype_drv_cmap_class =
{
    sizeof ( ITYPE_DRV_CMapRec ),
    itype_drv_cmap_init,
    itype_drv_cmap_done,
    itype_drv_cmap_char_index,
    itype_drv_cmap_char_next,

    /* Subsequent entries are special ones for format 14 -- the variant */
    /* selector subtable which behaves like no other                    */
    NULL, NULL, NULL, NULL, NULL,

    /* Subsequent entries match TT_CMap_ClassRec */
    0,
    itype_drv_cmap_validate,
    itype_drv_get_cmap_info
};

static const FT_CMap_Class ltt_cmap_classes[] =
{
    (const FT_CMap_Class)(void *)&itype_drv_cmap_class,
    NULL,
};






/*************************************************************************/
/*                                                                       */
/*                   itype_drv_convert_itype_error()                     */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_convert_itype_error( FS_LONG itype_err )
{
    FT_Error error = ITYPE_DRV_Err_Ok;

    /* file errors */
    if(( itype_err >= ERR_FILE_CREATE ) && ( itype_err <= ERR_FILE_WRITE ))
    {
        switch( itype_err )
        {
        case ERR_FILE_CREATE:
            error = ITYPE_DRV_Err_iType_File_Create_Error;
            break;
        case ERR_FILE_OPEN:
            error = ITYPE_DRV_Err_iType_File_Open_Error;
            break;
        case ERR_FILE_CLOSE:
            error = ITYPE_DRV_Err_iType_File_Close_Error;
            break;
        case ERR_FILE_SEEK:
            error = ITYPE_DRV_Err_iType_File_Seek_Error;
            break;
        case ERR_FILE_TELL:
            error = ITYPE_DRV_Err_iType_File_Tell_Error;
            break;
        case ERR_FILE_READ:
            error = ITYPE_DRV_Err_iType_File_Read_Error;
            break;
        case ERR_FILE_WRITE:
            error = ITYPE_DRV_Err_iType_File_Write_Error;
            break;
        default:
            error = ITYPE_DRV_Err_iType_Error;
            break;
        }

    /* memory errors */
    }
    else if(( itype_err >= ERR_MALLOC_FAIL ) && ( itype_err <= ERR_RESIZE_FAIL ))
    {
        switch( itype_err )
        {
        case ERR_MALLOC_FAIL:
            error = ITYPE_DRV_Err_iType_Memory_Malloc_Error;
            break;
        case ERR_REALLOC_FAIL:
            error = ITYPE_DRV_Err_iType_Memory_Realloc_Error;
            break;
        case ERR_FREE_BAD_PTR:
            error = ITYPE_DRV_Err_iType_Memory_Free_Bad_Pointer_Error;
            break;
        case ERR_RESIZE_FAIL:
            error = ITYPE_DRV_Err_iType_Memory_Resize_Error;
            break;
        default:
            error = ITYPE_DRV_Err_iType_Error;
            break;
        }

    /* SFNT/LFNT errors */
    }
    else if(( itype_err >= ERR_FONT_NOT_FOUND ) && ( itype_err <= ERR_FONT_NAME_IN_USE ))
    {
        error = ITYPE_DRV_Err_iType_SFNT_LFNT_Error;

    /* TT errors */
    }
    else if(( itype_err >= ERR_CMAP_UNSUPPORTED ) && ( itype_err <= ERR_BAD_LTT_INDEX ))
    {
        error = ITYPE_DRV_Err_iType_Truetype_Error;

    /* Random errors */
    }
    else if(( itype_err >= ERR_SCALE_LIMIT ) && ( itype_err < ERR_NOT_SUPPORTED ))
    {
        error = ITYPE_DRV_Err_iType_Random_Error;
    }
    else if ( itype_err == ERR_NOT_SUPPORTED )
    {
        error = ITYPE_DRV_Err_Invalid_Argument;
    }
    /* Other errors */
    else if(( itype_err >= ERR_TINY_TYPE ) && ( itype_err <= ERR_BAD_OUTLINE_OPACITY ))
    {
        error = ITYPE_DRV_Err_iType_Other_Error;

    /* Table Pointer errors */ 
    }
    else if(( itype_err >= ERR_TINY_TYPE ) && ( itype_err <= ERR_BAD_OUTLINE_OPACITY ))
    {
        error = ITYPE_DRV_Err_iType_Table_Pointer_Error;

    /* Mutex errors */
    }
    else if(( itype_err >= ERR_MUTEX_CREATE ) && ( itype_err <= ERR_MUTEX_RELEASE ))
    {
        error = ITYPE_DRV_Err_iType_Mutex_Error;

    /* Shared Memory errors */
    }
    else if(( itype_err >= ERR_SHMEM_CREATE ) && ( itype_err <= ERR_SHMEM_OPEN ))
    {
        error = ITYPE_DRV_Err_iType_Shared_Memory_Error;

    /* File Mapping errors */
    }
    else if(( itype_err >= ERR_FILE_UNMAP ) && ( itype_err <= ERR_MAP_FILE_VIEW ))
    {
        error = ITYPE_DRV_Err_iType_File_Mapping_Error;

    /* File Mapping errors */
    }
    else
    {
        error = ITYPE_DRV_Err_iType_Unknown_iType_Error;
    }
    return error;
}

/*************************************************************************/
/*                                                                       */
/*                          itype_drv_module_init()                      */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_module_init( FT_Module  module )
{
    FT_Error error = ITYPE_DRV_Err_Ok;
    FT_Memory memory;
    FS_LONG itype_err;
    FS_ULONG heap_size = ITYPE_HEAP_SIZE;
    Generic_Module_Data *GMD;

    if( module == NULL )
        return ITYPE_DRV_Err_Invalid_Argument;

    memory = FT_MODULE_MEMORY( module );
    if( memory == NULL )
        return ITYPE_DRV_Err_Invalid_Argument;

    if( FT_ALLOC( GMD, sizeof(Generic_Module_Data) ) ) {
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    memset( GMD, 0, sizeof(Generic_Module_Data) );

    if( FT_ALLOC( GMD->FS_state_ptr, sizeof(FS_STATE) ) ) {
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    /* initialize the iType font engine */
    itype_err = FS_init( GMD->FS_state_ptr, heap_size );
    if( itype_err != SUCCESS ) {
        FT_FREE( GMD->FS_state_ptr );
        FT_FREE( GMD );

        module->generic.finalizer = NULL;
        module->generic.data = NULL;

        error = ITYPE_DRV_Err_iType_Init_Failed;
    }
    else {
        module->generic.finalizer = NULL;
        module->generic.data = GMD;
    }
    return error;
}

/*************************************************************************/
/*                                                                       */
/*                            itype_drv_module_done()                    */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_drv_module_done( FT_Module  module )
{
    FS_STATE *FS_state_ptr;
    FT_Memory memory;
    Generic_Module_Data *GMD;

    if( module == NULL )
        return;

    GMD = module->generic.data;
    FS_state_ptr = GMD->FS_state_ptr;

    memory = FT_MODULE_MEMORY( module );
    if( memory == NULL )
        return;

    if( FS_state_ptr ) {
        /* always returns SUCCESS */
        FS_exit( FS_state_ptr );
    }

    FT_FREE( FS_state_ptr );
    FT_FREE( GMD );

    module->generic.finalizer = NULL;
    module->generic.data = NULL;
}

/*************************************************************************/
/*                                                                       */
/*                   itype_drv_module_get_interface()                    */
/*                                                                       */
/*************************************************************************/
FT_CALLBACK_DEF( FT_Module_Interface )
itype_drv_module_get_interface( FT_Module   driver,
                                const char* tt_interface )
{
    FT_Module            sfntd;
    SFNT_Service         sfnt;

    /* only return the default interface from the SFNT module */
    sfntd = FT_Get_Module( driver->library, "sfnt" );
    if( sfntd )
    {
      sfnt = (SFNT_Service)( sfntd->clazz->module_interface );
      if( sfnt )
        return sfnt->get_interface( driver, tt_interface );
    }
    return 0;
}

/*************************************************************************/
/*                                                                       */
/*                    itype_drv_create_face_client()                     */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FS_STATE * )
itype_drv_create_face_client( FT_Memory  memory, FS_STATE *root )
{
    FS_STATE *state;
    FT_Error  error;

    if( FT_ALLOC( state, sizeof(FS_STATE)) )
        return NULL;

    FT_MEM_SET( state, 0, sizeof(FS_STATE) );

    /* do state init here */
    state->platform = state->encoding = 0xFFFF;
    state->varPlatform = 0;
    state->varEncoding = 5;
    state->stroke_pct = FS_DEFAULT_STROKE_PCT;
    state->outline_width = 1;
    state->outline_opacity = (FS_FIXED)65536; /* 1.0 in 16.16 fixed point */

    state->parent = root;
    state->parent->ref_count++;
    state->heap_size = sizeof(FS_STATE);
    state->server = root->server;
    state->server->client_count++;

    /* To define custom default CSM values, change the values below */
#if defined FS_EDGE_TECH && defined FS_EDGE_RENDER
    FS_set_csm( state, (FS_FIXED) 26214  /*+0.4*/, -36700  /*-0.56*/, 65536  /*+1.00*/ );
#endif

    return state;
}

/*************************************************************************/
/*                                                                       */
/*                    itype_drv_destroy_face_client()                    */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FS_ULONG )
itype_drv_destroy_face_client( FT_Memory memory, FS_STATE *sp )
{
    sp->parent->ref_count--;
    sp->server->client_count--;

    /* clear the current typeset */
    TYPESET_destroy( sp, &sp->cur_typeset );

    FT_FREE( sp );

    return SUCCESS;
}

/*************************************************************************/
/*                                                                       */
/*                    itype_drv_create_mti_extension()                   */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_create_mti_extension(FT_Face ttface)
{
    FT_Error error;
    MTI_Info extension;
    Generic_Module_Data *GMD;
    FS_STATE *FS_state_ptr;
    FT_Memory memory = FT_FACE_MEMORY( ttface );

    if( ttface->extensions != NULL )
        return ITYPE_DRV_Err_Unknown_File_Format; /* something is wrong, the extenison should not be assgined yet */

    /* retrieve the root iType state */
    GMD = ttface->driver->root.generic.data;
    FS_state_ptr = GMD->FS_state_ptr;
    if( FS_state_ptr == NULL )
        return ITYPE_DRV_Err_Unknown_File_Format;

    /* allocate the extension */
    if( FT_ALLOC( extension, sizeof(MTI_InfoRec)) )
        return ITYPE_DRV_Err_Out_Of_Memory;

    FT_MEM_SET( extension, 0, sizeof(MTI_InfoRec) );
    extension->magic = ITYPE_EXT_MAGIC;

    extension->face_client = itype_drv_create_face_client( memory, FS_state_ptr );
    if( extension->face_client == NULL )
    {
        FT_FREE( extension );
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    /* assign it */
    ttface->extensions = extension;

    return ITYPE_DRV_Err_Ok;
}

/*************************************************************************/
/*                                                                       */
/*                    itype_drv_destroy_mti_extension()                  */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_destroy_mti_extension( FT_Face ttface )
{
    FT_Memory memory = FT_FACE_MEMORY( ttface );
    MTI_Info extension;

    if( !VALID_MTI_EXTENSION(ttface->extensions) )
        return ITYPE_DRV_Err_Unknown_File_Format;

    extension = (MTI_Info)(ttface->extensions);

    itype_drv_destroy_face_client( memory, extension->face_client );

    if( extension->font_name != NULL )
        FT_FREE( extension->font_name );
    FT_FREE( extension );
    ttface->extensions = NULL;

    return ITYPE_DRV_Err_Ok;
}






/*************************************************************************/
/*                                                                       */
/*                     itype_drv_validate_itype_size()                   */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_drv_validate_itype_size( FT_Size size )
{
    if( size )
    {
        if(( size->metrics.x_ppem > ITYPE_SCALE_UPPER_LIMIT ) ||
           ( size->metrics.y_ppem > ITYPE_SCALE_UPPER_LIMIT ))
        {
            size->metrics.x_ppem = ITYPE_SCALE_UPPER_LIMIT;
            size->metrics.y_ppem = ITYPE_SCALE_UPPER_LIMIT;
            size->metrics.x_scale = ( ITYPE_SCALE_UPPER_LIMIT * size->face->units_per_EM ) << 6;
            size->metrics.y_scale = ( ITYPE_SCALE_UPPER_LIMIT * size->face->units_per_EM ) << 6;
        }
        else if(( size->metrics.x_ppem < ITYPE_SCALE_LOWER_LIMIT ) ||
                  ( size->metrics.y_ppem < ITYPE_SCALE_LOWER_LIMIT ))
        {
            size->metrics.x_ppem = ITYPE_SCALE_LOWER_LIMIT;
            size->metrics.y_ppem = ITYPE_SCALE_LOWER_LIMIT;
            size->metrics.x_scale = ( ITYPE_SCALE_LOWER_LIMIT * size->face->units_per_EM ) << 6;
            size->metrics.y_scale = ( ITYPE_SCALE_LOWER_LIMIT * size->face->units_per_EM ) << 6;
        }
    }
}

/*************************************************************************/
/*                                                                       */
/*                       itype_drv_set_itype_size()                      */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FS_LONG )
itype_drv_set_itype_size( FT_Size size, FT_Int32 load_flags )
{
    FS_LONG itype_err = ERR_NOT_SUPPORTED;

    if( VALID_MTI_EXTENSION(size->face->extensions) )
    {
        MTI_Info ext = (MTI_Info)size->face->extensions;
        FS_STATE *FS_state_ptr = ext->face_client;

        if ( FS_state_ptr != NULL )
        {
            itype_drv_validate_itype_size( size );

            /* driver only load up-right glyphs, rotation will be apply to renderer */
            if( load_flags & FT_LOAD_NO_SCALE )
                itype_err = FS_set_scale( FS_state_ptr,
                                          (FS_FIXED) size->face->units_per_EM << 16,
                                          (FS_FIXED) 0,
                                          (FS_FIXED) 0,
                                          (FS_FIXED) size->face->units_per_EM << 16 );
            else
                itype_err = FS_set_scale( FS_state_ptr,
                                          (FS_FIXED) size->metrics.x_ppem << 16,
                                          (FS_FIXED) 0,
                                          (FS_FIXED) 0,
                                          (FS_FIXED) size->metrics.y_ppem << 16 );
        }
    }

    return itype_err;
}

/*************************************************************************/
/*                                                                       */
/*                      itype_drv_create_font_name()                     */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_drv_create_font_name( FT_Face ttface )
{
    if( VALID_MTI_EXTENSION(ttface->extensions) )
    {
        MTI_Info extension = (MTI_Info)(ttface->extensions);
        FT_Memory memory = FT_FACE_MEMORY( ttface );
        size_t len = 0;
        FT_Error error = ITYPE_DRV_Err_Ok;

        /*
         *  concatenate the family_name and the style_name together
         *  to create the font name
         */
        if( ttface->family_name != NULL )
        {
            len = ft_strlen( ttface->family_name ) + 2;
            if( ttface->style_name != NULL )
                len += ft_strlen( ttface->style_name );
        }

        if( len > 0 )
        {
            if( !FT_ALLOC( extension->font_name, len + 1 ) )
            {
                strcat( extension->font_name, ttface->family_name );
                if( ttface->style_name != NULL )
                {
                    strcat( extension->font_name, " " );
                    strcat( extension->font_name, ttface->style_name );
                }
            }
        }
    }
}

/*************************************************************************/
/*                                                                       */
/*                      itype_drv_get_font_name()                        */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FILECHAR * )
itype_drv_get_font_name( FT_Face ttface )
{
    if( VALID_MTI_EXTENSION(ttface->extensions) )
    {
        MTI_Info extension = (MTI_Info)(ttface->extensions);
        return extension->font_name;
    }

    return NULL;
}

/*************************************************************************/
/*                                                                       */
/*                     itype_drv_calc_num_of_itype_points()              */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Int )
itype_drv_calc_num_of_itype_points( FS_STATE *FS_state_ptr, FS_OUTLINE *outl )
{
    FT_Int n_points, i;
    FT_Int points;
    FS_BYTE *type;

    FT_UNUSED(FS_state_ptr);

    n_points = outl->num;
    type = outl->type;

    for (i = 0, points = 0; i < n_points; ++i, type++)
    {
        switch(*type & 0x0F)
        {
        case FS_MOVETO:
            points++;
            break;
        case FS_LINETO:
            points++;
            break;
        case FS_QUADTO:
            points++;
            points++;
            break;
        case FS_CUBETO:
            points++;
            points++;
            points++;
            break;
        default: 
            break;
        }
    }
    return points;
}

/*************************************************************************/
/*                                                                       */
/*                   itype_drv_fill_in_slot_outline()                    */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_fill_in_slot_outline( FT_GlyphSlot slot, FS_OUTLINE *outl, FT_Int32 load_flags )
{
    FT_Error error = ITYPE_DRV_Err_Ok;
    FS_STATE *FS_state_ptr;
    FS_FIXED *x,*y;
    FS_OUTLINE *itype_outline;
    FT_Outline *ft_outl;
    MTI_Info mti_ext;

    FT_Memory memory = FT_FACE_MEMORY( slot->face );

    /* check inputs */
    if( outl == NULL )
        return ITYPE_DRV_Err_Invalid_Argument;

    if( !VALID_MTI_EXTENSION(slot->face->extensions) )
        return ITYPE_DRV_Err_Invalid_Argument;

    mti_ext = (MTI_Info)slot->face->extensions;

    if( mti_ext->face_client == NULL )
        return ITYPE_DRV_Err_Invalid_Argument;

    /* get iType state */
    FS_state_ptr = mti_ext->face_client;

    /* clear any existing outline */
    if( mti_ext->outline != NULL )
    {
        FT_FREE( mti_ext->outline->points );
        FT_FREE( mti_ext->outline->tags );
        FT_FREE( mti_ext->outline->contours );
        FT_FREE( mti_ext->outline );
        mti_ext->outline = NULL;
    }

    slot->format = FT_GLYPH_FORMAT_OUTLINE;
    itype_outline = outl;

#ifdef DUMP_OUTLINES
    itype_drv_dump_itype_outline( itype_outline, "returned by FS_get_outline()." );
#endif
    if( FT_ALLOC( ft_outl, sizeof(FT_Outline) ))
    {
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    ft_outl->n_contours = itype_outline->nc;
    ft_outl->n_points = (short)(itype_drv_calc_num_of_itype_points( FS_state_ptr, itype_outline));
    ft_outl->n_points -= ft_outl->n_contours; /* compensate for end points */

    if( ft_outl->n_points > 0 )
    {
        FT_UInt n_types = itype_outline->num;
        FT_UInt i, j, k, l, n_points = ft_outl->n_points;
        FS_FIXED round = (1L << 9); /* for rounding points */

        /* Allocate space for the points PLUS the raster params we hide at the end */
        if( FT_ALLOC( ft_outl->points,(sizeof(FT_Vector) * ft_outl->n_points)+(sizeof(FT_Raster_User_Params_ITC)) ))
        {
            FT_FREE( ft_outl );
            return ITYPE_DRV_Err_Out_Of_Memory;
        }
        if( FT_ALLOC( ft_outl->tags, sizeof(char) * ft_outl->n_points ))
        {
            FT_FREE( ft_outl->points );
            FT_FREE( ft_outl );
            return ITYPE_DRV_Err_Out_Of_Memory;
        }
        if( FT_ALLOC( ft_outl->contours, sizeof(short) * ft_outl->n_contours ))
        {
            FT_FREE( ft_outl->tags );
            FT_FREE( ft_outl->points );
            FT_FREE( ft_outl );
            return ITYPE_DRV_Err_Out_Of_Memory;
        }

        /* since not a space character, convert the data (except for end points) */
        /* i = iType type index,     j = iType point index,        */
        /* k = FreeType point index, l = FreeType contour index    */
        x = itype_outline->x;
        y = itype_outline->y;
        for( i = 0, j = 0, k = 0, l = 0; i < n_types; i++ )
        {
            FT_Char type = itype_outline->type[i] & 0x7f;
            switch( type )
            {
            case FS_MOVETO:
                if(k > 1)
                {
                    k--; /* overwrite previous closed point */
                    ft_outl->contours[l] = (short)(k-1);
                    l++;
                }
                ft_outl->points[k].x = (x[j]+round) >> 10;
                ft_outl->points[k].y = (y[j]+round) >> 10;
                ft_outl->tags[k] = FT_CURVE_TAG_ON;
                j++; k++;
                break;

            case FS_LINETO:
                if( k == n_points )
                    break; /* ignore last closed point */

                ft_outl->points[k].x = (x[j]+round) >> 10;
                ft_outl->points[k].y = (y[j]+round) >> 10;
                ft_outl->tags[k] = FT_CURVE_TAG_ON;
                j++; k++;
                break;

            case FS_QUADTO:
                ft_outl->points[k].x = (x[j]+round) >> 10;
                ft_outl->points[k].y = (y[j]+round) >> 10;
                ft_outl->tags[k] = FT_CURVE_TAG_CONIC;
                j++; k++;
                if( k == n_points )
                    break; /* ignore last closed point */

                ft_outl->points[k].x = (x[j]+round) >> 10;
                ft_outl->points[k].y = (y[j]+round) >> 10;
                ft_outl->tags[k] = FT_CURVE_TAG_ON;
                j++; k++;
                break;

            case FS_CUBETO:
                ft_outl->points[k].x = (x[j]+round) >> 10;
                ft_outl->points[k].y = (y[j]+round) >> 10;
                ft_outl->tags[k] = FT_CURVE_TAG_CUBIC;
                j++; k++;

                ft_outl->points[k].x = (x[j]+round) >> 10;
                ft_outl->points[k].y = (y[j]+round) >> 10;
                ft_outl->tags[k] = FT_CURVE_TAG_CUBIC;
                j++; k++;
                if( k == n_points )
                    break; /* ignore last closed point */

                ft_outl->points[k].x = (x[j]+round) >> 10;
                ft_outl->points[k].y = (y[j]+round) >> 10;
                ft_outl->tags[k] = FT_CURVE_TAG_ON;
                j++; k++;
                break;

            default:
                break;
            }
        }
        /* handle case where n_contours is 1 */
        ft_outl->contours[l] = (short)(ft_outl->n_points-1);
    }


#ifdef DUMP_OUTLINES
    /* print the non close-point ft_outline */
    itype_drv_dump_freetype_outline( ft_outl, "ft_outline - now with no-close-points." );
#endif

    if ( !(itype_outline->outl_flag & OUTL_FLAGS_SCANCTRL) )
        ft_outl->flags |= FT_OUTLINE_IGNORE_DROPOUTS;

    /* outline is in design_units when FFT_LOAD_NO_SCALE */ 
    if( load_flags & FT_LOAD_NO_SCALE )
    {
        int i;
        for( i=0; i<ft_outl->n_points; i++ )
        {
            ft_outl->points[i].x = (ft_outl->points[i].x + 32) >> 6;
            ft_outl->points[i].y = (ft_outl->points[i].y + 32) >> 6;
        }
    }

    /* load the new none close-points outline to the slot */
    slot->outline.n_points     = ft_outl->n_points;
    slot->outline.n_contours   = ft_outl->n_contours;
    slot->outline.points       = ft_outl->points;
    slot->outline.tags         = ft_outl->tags;
    slot->outline.contours     = ft_outl->contours;
    slot->outline.flags        = ft_outl->flags | FT_OUTLINE_FROM_ITYPE_DRV;

    mti_ext->outline = ft_outl;

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                       itype_drv_add_raster_params                     */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_drv_add_raster_params( FT_GlyphSlot slot, FS_STATE *FS_state_ptr,
                             FT_UInt glyph_index )
{
    FT_Raster_User_Params_ITC *params;

    if( slot->outline.n_points == 0 )
        return;

    /* Get pointer to the raster params which are appended to the outline
       points and were allocated in itype_drv_fill_in_slot_outline */
    params = (FT_Raster_User_Params_ITC *)(void *)&slot->outline.points[slot->outline.n_points];

    /* set default values */
    params->magic = 1234;
    params->ppem = (short)FS_state_ptr->lpm;
    params->bitmap_left = 0;
    params->bitmap_top = 0;
    params->renderWithEdge = 0/*FALSE*/;
    params->gammaValue = FIXED_ONE;
    params->insideCutoff  = 0;
    params->outsideCutoff = 0;
    params->sharpness_offset = 0;
    params->sharpness_slope = 0;
    params->thickness_offset = 0;
    params->thickness_slope = 0;

#ifdef FS_EDGE_RENDER

#ifdef ITYPE_RENDER_EDGE_ALWAYS

    /* always set the flag */
    params->renderWithEdge = 1/*TRUE*/;

#elif defined ITYPE_RENDER_EDGE_IF_ADFH_PRESENT
    {
    TTF *ttf = (TTF *)(FS_state_ptr->cur_lfnt->fnt);

    if ( ( ttf != NULL ) && (ttf->adfh != NULL) )
        params->renderWithEdge = 1/*TRUE*/;
    }
#endif

    /* get the csms. This will get the default values if the font is not an Edge font */
    FS_get_csm( FS_state_ptr, (FS_USHORT)glyph_index,
                (FS_FIXED *)&params->insideCutoff,
                (FS_FIXED *)&params->outsideCutoff,
                (FS_FIXED *)&params->gammaValue );

    /* get the adjustments */
    if( VALID_MTI_EXTENSION(slot->face->extensions) )
    {
        MTI_Info ext = (MTI_Info)slot->face->extensions;

        params->sharpness_offset = ext->sharpness_offset;
        params->sharpness_slope = ext->sharpness_slope;
        params->thickness_offset = ext->thickness_offset;
        params->thickness_slope = ext->thickness_slope;
    }
    else
    {
        ITC_PRINTF_DEBUG(("itype_drv_add_raster_params -- invalid face extension!!\n"));
    }

#ifdef TXTC_BUILDS
    {
        TTF *ttf = (TTF *)(FS_state_ptr->cur_lfnt->fnt);
        if ( ( ttf != NULL ) && (ttf->adfh != NULL) )
        {
            if ( strncmp( FS_state_ptr->cur_typeset.fntset->name, "givemeinfo", 20 ) == 0 ) /* test font */
                params->renderWithEdge = 2;
        }
    }
#endif

#else
    glyph_index = glyph_index; /* avoid compiler warning */
#endif
}

/*************************************************************************/
/*                                                                       */
/*                     itype_drv_fill_in_slot_metrics()                  */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_fill_in_slot_metrics( FT_GlyphSlot slot,
                                FS_STATE    *FS_state_ptr,
                                FS_OUTLINE  *outl,
                                FT_Int32     load_flags,
                                FS_USHORT    glyph_index )
{
    FT_Short lsb, tsb, aw, ah;
    FT_Short scaled_aw, scaled_ah;
    FS_USHORT unitsPerEm;
    FT_Error error = ITYPE_DRV_Err_Ok;
    TTF *ttf;
    FS_FIXED lo_x;
    FS_FIXED hi_x;
    FS_FIXED lo_y;
    FS_FIXED hi_y;

    if( slot == NULL )
        return ITYPE_DRV_Err_Invalid_Slot_Handle;

    if( outl == NULL )
        return ITYPE_DRV_Err_Invalid_Argument;

    ttf = FS_state_ptr->cur_typeset.tfntarray[FS_state_ptr->cur_font].cfnt->lfnt->fnt;
    unitsPerEm = ttf->head->unitsPerEm;
    FS_get_glyph_metrics(FS_state_ptr, glyph_index,&lsb,&aw,&tsb,&ah);

    /* in linked font a glyph from component font may have diff units_per_EM than metrics font */
    /* we need to scale in this case                                                            */
    if( unitsPerEm != slot->face->units_per_EM )
    {
        aw = aw * slot->face->units_per_EM / unitsPerEm;
        ah = ah * slot->face->units_per_EM / unitsPerEm;
    }
    scaled_aw = (aw * slot->face->size->metrics.x_scale)>>16;
    scaled_ah = (ah * slot->face->size->metrics.y_scale)>>16;

    hi_x = outl->hi_x;
    hi_y = outl->hi_y;
    lo_x = outl->lo_x;
    lo_y = outl->lo_y;

    /* adjust the size of the outline if rendering with edge and CSMs will result in ink outside of the outline */
    {
    FT_Raster_User_Params_ITC *params = NULL;

    if( slot->outline.n_points != 0)
        params = (FT_Raster_User_Params_ITC *)&slot->outline.points[slot->outline.n_points];

    if( (params != NULL) && (params->magic == 1234) && (params->renderWithEdge == 1/*TRUE*/) )
    {
        FT_Fixed S, T, adjustedOutsideCutoff;

        S = (params->sharpness_offset + params->ppem * params->sharpness_slope);
        T = (params->thickness_offset + params->ppem * params->thickness_slope);

        adjustedOutsideCutoff = params->outsideCutoff + (S - T);

        if ((int)adjustedOutsideCutoff < 0L) /* extends beyond outline */
        {
            FT_Pos delta = ((-(int)adjustedOutsideCutoff)+0xffff)&0xffff0000;

            hi_x += delta;
            hi_y += delta;
            lo_x -= delta;
            lo_y -= delta;
        }
    }
    }

    /* if FT_LOAD_NO_SCALE is set then return values are in font units
     * Otherwise return values are in 26.6 format iType outl->hi_x is in 16.16
     */
    if( load_flags & FT_LOAD_NO_SCALE )
    {
        /* metrics are in integer font units */
        slot->metrics.width  = (hi_x - lo_x + (1L << 15)) >> 16;
        slot->metrics.height = (hi_y - lo_y + (1L << 15)) >> 16;

        if ( (load_flags & FT_LOAD_VERTICAL_LAYOUT) )
        {
            slot->metrics.vertBearingX = (lo_x + (1L << 15)) >> 16;
            slot->metrics.vertBearingY = (hi_y + (1L << 15)) >> 16;

            slot->metrics.horiAdvance  = aw;
            slot->metrics.vertAdvance  = (FS_FIXED)outl->i_dx;

            slot->metrics.horiBearingX = slot->metrics.vertBearingX + 1 - (ah>>1);
            slot->metrics.horiBearingY = slot->metrics.horiAdvance - slot->metrics.vertBearingY;
        }
        else if ( (load_flags & FT_LOAD_VERTICAL_ROTATE_RIGHT) || (load_flags & FT_LOAD_VERTICAL_ROTATE_LEFT))
        {
            slot->metrics.horiBearingX = (lo_x + (1L << 15)) >> 16;
            slot->metrics.horiBearingY = (hi_y + (1L << 15)) >> 16;

            slot->metrics.vertBearingX =  (lo_x + (1L << 15)) >> 16;
            slot->metrics.vertBearingY = -((hi_y + (1L << 15)) >> 16);

            slot->metrics.horiAdvance  = aw;
            slot->metrics.vertAdvance  = (FS_FIXED)outl->i_dy;
        }
        else
        {
            slot->metrics.horiBearingX = (lo_x + (1L << 15)) >> 16;
            slot->metrics.horiBearingY = (hi_y + (1L << 15)) >> 16;
            
            slot->metrics.horiAdvance  = (FS_FIXED)outl->i_dx;
            slot->metrics.vertAdvance  = ah;

            slot->metrics.vertBearingX = slot->metrics.horiBearingX + 1 - (aw>>1);
            slot->metrics.vertBearingY = slot->metrics.vertAdvance - slot->metrics.horiBearingY;;
        }
    }
    else
    {
        /* metrics are in 26.6 pixels */
        slot->metrics.width  = FIXEDTODOT6(hi_x - lo_x);
        slot->metrics.height = FIXEDTODOT6(hi_y - lo_y);

        if ( (load_flags & FT_LOAD_VERTICAL_LAYOUT) )
        {
            slot->metrics.vertBearingX =   FIXEDTODOT6(lo_x)  + 64;
            slot->metrics.vertBearingY = -(FIXEDTODOT6(hi_y)) + 64;

            slot->metrics.horiAdvance  = FT_PIX_ROUND(scaled_aw);
            slot->metrics.vertAdvance  = ((FS_FIXED)outl->i_dy) << 6;

            slot->metrics.horiBearingX = slot->metrics.vertBearingX + (scaled_aw>>1) - 64;
            slot->metrics.horiBearingY = slot->metrics.vertAdvance - slot->metrics.vertBearingY + 64;
        }
        else if ( (load_flags & FT_LOAD_VERTICAL_ROTATE_RIGHT) || (load_flags & FT_LOAD_VERTICAL_ROTATE_LEFT))
        {
            slot->metrics.horiBearingX = FIXEDTODOT6(lo_x);
            slot->metrics.horiBearingY = FIXEDTODOT6(hi_y);

            slot->metrics.vertBearingX = FIXEDTODOT6(lo_x);
            slot->metrics.vertBearingY = -(FIXEDTODOT6(hi_y));

            slot->metrics.horiAdvance  = FT_PIX_ROUND(scaled_ah);
            slot->metrics.vertAdvance  = ((FS_FIXED)outl->i_dy) << 6;
        }
        else
        {
            slot->metrics.horiBearingX = FIXEDTODOT6(lo_x);
            slot->metrics.horiBearingY = FIXEDTODOT6(hi_y);
            
            slot->metrics.horiAdvance  = ((FS_FIXED)outl->i_dx) << 6;
            slot->metrics.vertAdvance  = FT_PIX_ROUND(scaled_ah);

            slot->metrics.vertBearingX = slot->metrics.horiBearingX - (scaled_aw>>1);
            slot->metrics.vertBearingY = slot->metrics.vertAdvance - slot->metrics.horiBearingY;
        }
    }

    slot->linearHoriAdvance = aw;
    slot->linearVertAdvance = ah;

    slot->advance.x = ((FS_FIXED)outl->i_dx) << 6; /* integer to 26.6 */
    slot->advance.y = ((FS_FIXED)outl->i_dy) << 6;

    /* itc sets vertical metrics correctly, so we mark it */
    slot->face->face_flags |= FT_FACE_FLAG_VERTICAL;

    return error;
 }



/*************************************************************************/
/*                                                                       */
/*                       itype_drv_load_kern_in_face()                   */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_load_kern_in_face( TT_Face    face)
{
    FT_Error   error = ITYPE_DRV_Err_Ok;
    FT_Byte*   p;
    FT_Byte*   p_limit;
    FT_UInt    nn, num_tables;
    FT_UInt32  avail = 0, ordered = 0;

    if( VALID_MTI_EXTENSION(face->root.extensions) )
    {
        FS_STATE *FS_state_ptr;
        FS_LONG itype_err;
        FS_ULONG table_size = 0;

        MTI_Info ext = (MTI_Info)face->root.extensions;

        FS_state_ptr = ext->face_client;
        itype_err = FS_set_font(FS_state_ptr, ext->font_name);
        if( itype_err == SUCCESS )
        {
            face->kern_table = (FT_Byte *)FS_get_table(FS_state_ptr, TAG_kern, TBL_EXTRACT, &table_size);
            if(face->kern_table == NULL)
            {
                return ITYPE_DRV_Err_Ok;
            }

            if ( table_size < 4 )  /* the case of a malformed table */
            {
                FT_ERROR(( "ltt_face_load_kern:"
                            " kerning table is too small - ignored\n" ));
                error = FT_THROW( Table_Missing );
                return error;
            }

            face->kern_table_size = (FT_ULong)table_size;

            p       = face->kern_table;
            p_limit = p + table_size;

            p         += 2; /* skip version */
            num_tables = FT_NEXT_USHORT( p );

            if ( num_tables > 32 ) /* we only support up to 32 sub-tables */
                num_tables = 32;

            for ( nn = 0; nn < num_tables; nn++ )
            {
                FT_UInt    num_pairs, length, coverage;
                FT_Byte*   p_next;
                FT_UInt32  mask = (FT_UInt32)1UL << nn;


                if ( p + 6 > p_limit )
                    break;

                p_next = p;

                p += 2; /* skip version */
                length   = FT_NEXT_USHORT( p );
                coverage = FT_NEXT_USHORT( p );

                if ( length <= 6 )
                    break;

                p_next += length;

                if ( p_next > p_limit )  /* handle broken table */
                    p_next = p_limit;

                /* only use horizontal kerning tables */
                if ( ( coverage & ~8 ) != 0x0001 ||
                    p + 8 > p_limit             )
                {
                    p = p_next;
                    continue;
                }

                num_pairs = FT_NEXT_USHORT( p );
                p        += 6;

                if ( ( p_next - p ) < 6 * (int)num_pairs ) /* handle broken count */
                    num_pairs = (FT_UInt)( ( p_next - p ) / 6 );

                avail |= mask;

                /*
                * Now check whether the pairs in this table are ordered.
                * We then can use binary search.
                */
                if ( num_pairs > 0 )
                {
                    FT_ULong  count;
                    FT_ULong  old_pair;


                    old_pair = FT_NEXT_ULONG( p );
                    p       += 2;

                    for ( count = num_pairs - 1; count > 0; count-- )
                    {
                        FT_UInt32  cur_pair;


                        cur_pair = FT_NEXT_ULONG( p );
                        if ( cur_pair <= old_pair )
                            break;

                        p += 2;
                        old_pair = cur_pair;
                    }

                    if ( count == 0 )
                        ordered |= mask;
                }

            }

            face->num_kern_tables = nn;
            face->kern_avail_bits = avail;
            face->kern_order_bits = ordered;
        }
    }

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                     itype_drv_load_ltt_info()                         */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_load_ltt_info( FT_Face ttface, FS_STATE *FS_state_ptr )
{
    FT_Error  error;
    FS_LONG   itype_err;
    FT_Memory memory = FT_FACE_MEMORY( ttface );
    TT_Face   face = (TT_Face)ttface;
    FONT_METRICS *metrics;
    TTF_HHEA *hhea;
    TTF_VHEA *vhea;
    TTF_POST *post;
    TTF_MAXP *maxp;
    TTF_HEAD *head;
    FS_ULONG  len;
    FT_CMap   cmap;
    TT_TableRec*    entry;
    const FT_CMap_Class* volatile  pclazz = ltt_cmap_classes;
    /*lint -fqb reverse the message: Note 963 */
    FT_CMap_Class volatile         clazz;
    /*lint +fqb reverse the message: Note 963 */

    /* load metrics data into face structure for ltt font */

    /* load the cmap access functions */
    clazz = *pclazz;

    if( FT_ALLOC( cmap, clazz->size ) )
        return ITYPE_DRV_Err_Out_Of_Memory;

    cmap->clazz = (FT_CMap_Class)clazz;
    if( FT_ALLOC( ttface->charmaps, sizeof(FT_CharMap)) )
    {
        FT_FREE( cmap );
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    ttface->num_charmaps = 1; /* LTT only has one, either 3,1 or 3,10 */
    ttface->charmaps[0] = (FT_CharMap)cmap;
    ttface->charmaps[0]->face = ttface;

    /* query font for supported cmap */
    FS_set_cmap(FS_state_ptr,3,10);
    if( FS_state_ptr->error )
    {
        FS_set_cmap(FS_state_ptr,3,1);
        if( FS_state_ptr->error )
        {
            FT_FREE( cmap );
            FT_FREE( ttface->charmaps );
            return ITYPE_DRV_Err_Unknown_File_Format;
        }
        else
        {
            ttface->charmaps[0]->encoding    = FT_ENCODING_UNICODE;
            ttface->charmaps[0]->platform_id = TT_PLATFORM_MICROSOFT;  /* 3 */
            ttface->charmaps[0]->encoding_id = TT_MS_ID_UNICODE_CS;    /* 1 */
        }
    }
    else
    {
        ttface->charmaps[0]->encoding    = FT_ENCODING_UNICODE;
        ttface->charmaps[0]->platform_id = TT_PLATFORM_MICROSOFT;  /* 3 */
        ttface->charmaps[0]->encoding_id = TT_MS_ID_UCS_4;         /* 10 */
    }

    if( FT_ALLOC( metrics, sizeof(FONT_METRICS) ))
    {
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    itype_err = FS_font_metrics( FS_state_ptr, metrics );
    if( itype_err  != SUCCESS )
    {
        FT_FREE( metrics );
        return itype_drv_convert_itype_error( itype_err );
    }

    face->os2.fsSelection = metrics->os2_fsSelection;
    ttface->bbox.xMin = metrics->font_bbox.xMin;
    ttface->bbox.yMin = metrics->font_bbox.yMin;
    ttface->bbox.xMax = metrics->font_bbox.xMax;
    ttface->bbox.yMax = metrics->font_bbox.yMax;
    ttface->units_per_EM = metrics->unitsPerEm;

    ttface->ascender  = metrics->hhea_ascent;
    ttface->descender = metrics->hhea_descent;
    ttface->height    = (FT_Short)( metrics->hhea_ascent - metrics->hhea_descent +
                                    metrics->hhea_leading );
    FT_FREE( metrics );

    /* Get "head" table */
    head = (TTF_HEAD *)(void *)FS_get_table( FS_state_ptr, TAG_head, TBL_EXTRACT, &len );
    if( head && FS_state_ptr->error == SUCCESS )
    {
        face->header.Table_Version       = SWAPL( head->version );
        face->header.Font_Revision       = SWAPL( head->fontRevision );
        face->header.CheckSum_Adjust     = SWAPL( head->checkSumAdjustment );
        face->header.Magic_Number        = SWAPL( head->magicNumber );
        face->header.Flags               = SWAPW( head->flags );
        face->header.Units_Per_EM        = SWAPW( head->unitsPerEm );
        face->header.Created[0]          = SWAPL( head->created[0] );
        face->header.Created[1]          = SWAPL( head->created[1] );
        face->header.Modified[0]         = SWAPL( head->modified[0] );
        face->header.Modified[1]         = SWAPL( head->modified[1] );
        face->header.xMin                = SWAPW( head->xMin );
        face->header.yMin                = SWAPW( head->yMin );
        face->header.xMax                = SWAPW( head->xMax );
        face->header.yMax                = SWAPW( head->yMax );
        face->header.Mac_Style           = SWAPW( head->macStyle );
        face->header.Lowest_Rec_PPEM     = SWAPW( head->lowestRecPPEM );
        face->header.Font_Direction      = SWAPW( head->fontDirectionHint );
        face->header.Index_To_Loc_Format = SWAPW( head->indexToLocFormat );
        face->header.Glyph_Data_Format   = SWAPW( head->glyphDataFormat );

        itype_err = FS_free_table( FS_state_ptr, head );
    }
    else
    {
        /* head table is required, so error if cannot get */
        return itype_drv_convert_itype_error( FS_state_ptr->error );
    }

    /* Get "maxp" table */
    maxp = (TTF_MAXP *)(void *)FS_get_table( FS_state_ptr, TAG_maxp,  TBL_EXTRACT, &len );
    if( maxp && FS_state_ptr->error == SUCCESS )
    {
        face->max_profile.version               = SWAPL( maxp->version );
        face->max_profile.numGlyphs             = SWAPW( maxp->numGlyphs );
        face->max_profile.maxPoints             = SWAPW( maxp->maxPoints );
        face->max_profile.maxContours           = SWAPW( maxp->maxContours );
        face->max_profile.maxCompositePoints    = SWAPW( maxp->maxCompositePoints );
        face->max_profile.maxCompositeContours  = SWAPW( maxp->maxCompositeContours );
        face->max_profile.maxZones              = SWAPW( maxp->maxElements ); /* ??? */
        face->max_profile.maxTwilightPoints     = SWAPW( maxp->maxTwilightPoints );
        face->max_profile.maxStorage            = SWAPW( maxp->maxStorage );
        face->max_profile.maxFunctionDefs       = SWAPW( maxp->maxFunctionDefs );
        face->max_profile.maxInstructionDefs    = SWAPW( maxp->maxInstructionDefs );
        face->max_profile.maxStackElements      = SWAPW( maxp->maxStackElements );
        face->max_profile.maxSizeOfInstructions = SWAPW( maxp->maxSizeOfInstructions );
        face->max_profile.maxComponentElements  = SWAPW( maxp->maxComponentElements );
        face->max_profile.maxComponentDepth     = SWAPW( maxp->maxComponentDepth );

        ttface->num_glyphs = face->max_profile.numGlyphs;
        itype_err = FS_free_table( FS_state_ptr, maxp );
    }
    else
    {   /* maxp table is required, so error if cannot get */
        return itype_drv_convert_itype_error( FS_state_ptr->error );
    }

    /* Get horizontal, vertical, and postScript tables: */
    if( FT_ALLOC( hhea, sizeof(TTF_HHEA) ) )
    {
        return ITYPE_DRV_Err_Out_Of_Memory;
    }
    else
    {
        itype_err = FS_get_table_structure( FS_state_ptr, TAG_hhea, hhea );
        if( itype_err == SUCCESS )
            ttface->max_advance_width  = hhea->advanceWidthMax;
        else
        {
            FT_FREE( hhea );
            return itype_drv_convert_itype_error( itype_err );
        }
        FT_FREE( hhea );
    }

    if( FT_ALLOC( vhea, sizeof(TTF_VHEA) ) )
    {
        return ITYPE_DRV_Err_Out_Of_Memory;
    }
    else
    {
        vhea->version = 0;
        itype_err = FS_get_table_structure( FS_state_ptr, TAG_vhea, vhea );

        /* the vhea table is not required so just continue regardless */
        ttface->max_advance_height = (FT_Short)( (itype_err == SUCCESS && vhea->version)
                                           ? vhea->advanceHeightMax
                                           : ttface->height );
        FT_FREE( vhea );
    }

    /* the post table is not required so just continue regardless */
    post = (TTF_POST *)(void *)FS_get_table( FS_state_ptr, TAG_post, TBL_EXTRACT, &len );
    if( post && FS_state_ptr->error == SUCCESS )
    {
        FS_SHORT position  = post->underlinePosition;
        FS_SHORT thickness = post->underlineThickness;
        FS_SHORT fixedpitch= (FS_SHORT)post->isFixedPitch;

        position  = SWAPW(position);
        thickness = SWAPW(thickness); 
        fixedpitch = SWAPW(fixedpitch);

        ttface->underline_position  = position - thickness / 2;
        ttface->underline_thickness = thickness;
        face->postscript.isFixedPitch = fixedpitch;

        /* since iType post table structure doesn't have data item 'FormatType', and */
        /* also we know our LTT font post table's FormatType always is 0x00030000L */
        face->postscript.FormatType = 0x00030000L;

        FS_free_table( FS_state_ptr, post );
    }

     /* LTT font doesn't have merged hmtx or vmtx table, so face->vertical_info will be set to 0 */
    face->vertical_info = 0;

    ttface->face_flags |= FT_FACE_FLAG_SCALABLE    |
                         FT_FACE_FLAG_HORIZONTAL  |
                         FT_FACE_FLAG_GLYPH_NAMES |
                         FT_FACE_FLAG_KERNING;

    /* fixed width font? */
    if( face->postscript.isFixedPitch )
        ttface->face_flags |= FT_FACE_FLAG_FIXED_WIDTH;

    /* since LTT font doesn't have merged hmtx or vmtx table, so face->vertical_info will be */
    /* left as 0 that will cause face_flags doesn't set FT_FACE_FLAG_VERTICAL */
    if( face->vertical_info )
        ttface->face_flags |= FT_FACE_FLAG_VERTICAL;

    /*********************************************************************/
    /*                                                                   */
    /*   Compute style flags.                                            */
    /*                                                                   */
    ttface->style_flags = 0;

    /* We have an OS/2 table; use the `fsSelection' field.  Bit 9 */
    /* indicates an oblique font face.  This flag has been        */
    /* introduced in version 1.5 of the OpenType specification.   */
    if( face->os2.fsSelection & 512 )       /* bit 9 */
        ttface->style_flags |= FT_STYLE_FLAG_ITALIC;
    else if( face->os2.fsSelection & 1 )    /* bit 0 */
        ttface->style_flags |= FT_STYLE_FLAG_ITALIC;

    if ( face->os2.fsSelection & 32 )        /* bit 5 */
        ttface->style_flags |= FT_STYLE_FLAG_BOLD;

    error = itype_drv_load_kern_in_face( face );

    if (FS_state_ptr->cur_typeset.fntset == NULL)
        return itype_drv_convert_itype_error(ERR_NO_CURRENT_FNTSET);

    face->num_tables = 0;
    if(FS_state_ptr->cur_typeset.fntset->gpos)
        face->num_tables++;
    if(FS_state_ptr->cur_typeset.fntset->gsub)
        face->num_tables++;
    if(FS_state_ptr->cur_typeset.fntset->gdef)
        face->num_tables++;
    if(FS_state_ptr->cur_typeset.fntset->name_offset)
        face->num_tables++;
    if(FS_state_ptr->cur_typeset.fntset->cmap)
        face->num_tables++;

    if ( FT_QNEW_ARRAY( face->dir_tables, face->num_tables ) )
        return ITYPE_DRV_Err_Out_Of_Memory;

    entry = face->dir_tables;
    if(FS_state_ptr->cur_typeset.fntset->gpos){
        entry->Tag      = TAG_GPOS;
        entry->CheckSum = FS_state_ptr->cur_typeset.fntset->checksum;
        entry->Offset   = FS_state_ptr->cur_typeset.fntset->gpos->offset;
        entry->Length   = FS_state_ptr->cur_typeset.fntset->gpos->size;
        entry++;
    }
    if(FS_state_ptr->cur_typeset.fntset->gsub){
        entry->Tag      = TAG_GSUB;
        entry->CheckSum = FS_state_ptr->cur_typeset.fntset->checksum;
        entry->Offset   = FS_state_ptr->cur_typeset.fntset->gsub->offset;
        entry->Length   = FS_state_ptr->cur_typeset.fntset->gsub->size;
        entry++;
    }
    if(FS_state_ptr->cur_typeset.fntset->gdef){
        entry->Tag      = TAG_GDEF;
        entry->CheckSum = FS_state_ptr->cur_typeset.fntset->checksum;
        entry->Offset   = FS_state_ptr->cur_typeset.fntset->gdef->offset;
        entry->Length   = FS_state_ptr->cur_typeset.fntset->gdef->size;
        entry++;
    }
    if(FS_state_ptr->cur_typeset.fntset->name_offset){
        entry->Tag      = TAG_name;
        entry->CheckSum = FS_state_ptr->cur_typeset.fntset->checksum;
        entry->Offset   = FS_state_ptr->cur_typeset.fntset->name_offset;
        entry->Length   = FS_state_ptr->cur_typeset.fntset->name_size;
        entry++;
    }
    if(FS_state_ptr->cur_typeset.fntset->cmap){
        entry->Tag      = TAG_cmap;
        entry->CheckSum = FS_state_ptr->cur_typeset.fntset->checksum;
        entry->Offset   = FS_state_ptr->cur_typeset.fntset->cmap->offset;
        entry->Length   = FS_state_ptr->cur_typeset.fntset->cmap->size;
    }

    return error;
}


/*************************************************************************/
/*                                                                       */
/*                         itype_drv_face_init()                         */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_face_init( FT_Stream      stream,
                     FT_Face        ttface,
                     FT_Int         face_index,
                     FT_Int         num_params,
                     FT_Parameter*  params )
{
    TT_Face    face = (TT_Face)ttface;
    FT_Error   error;
    FT_Library library = face->root.driver->root.library;
    FS_STATE  *FS_state_ptr;
    FS_LONG    itype_err;
    FT_ULong   format_tag;
    FT_Memory  memory = FT_FACE_MEMORY( ttface );
    FS_BOOLEAN bIsLttFont = FALSE;
    FS_BOOLEAN bIsLttDirFont = FALSE;
    SFNT_Service sfnt;
    char sStream[35];
    FS_FILE *fpLtt;
    FS_USHORT ix;

    /* create the extension (and iType client state) */
    error = itype_drv_create_mti_extension( ttface );
    if( error != FT_Err_Ok )
        return error;

    /* get the client iType state */
    FS_state_ptr = ((MTI_Info)(ttface->extensions))->face_client;
    if( FS_state_ptr == NULL )
        return ITYPE_DRV_Err_Unknown_File_Format;

    /* create input stream from resource */
    if( FT_STREAM_SEEK( 0 ) )
        return ITYPE_DRV_Err_Unknown_File_Format;

    if( FT_READ_ULONG(format_tag) )
        return ITYPE_DRV_Err_Unknown_File_Format;

    /* move offset back to beginning */
    if( FT_STREAM_SEEK( 0 ))
        return ITYPE_DRV_Err_Unknown_File_Format;

    /* create a "magic" name - prefix + stream pointer value */
    idrv_make_path( sStream, (long)stream );

    sfnt = (SFNT_Service)FT_Get_Module_Interface( library, "sfnt" );
    if( !sfnt )
    {
        itype_drv_destroy_mti_extension( ttface );
        return ITYPE_DRV_Err_Unknown_File_Format;
    }

    /* check for non-ltt fonts */
    if( format_tag != 0x4c54540a && format_tag != 0x4c54540d )
    {

        /* check that we have a valid TrueType file */
        error = sfnt->init_face( stream, face, face_index, num_params, params );
        if( error )
        {
            itype_drv_destroy_mti_extension( ttface );
            return error;
        }

        /* We must also be able to accept Mac/GX fonts, as well as OT ones. */
        /* The 0x00020000 tag is completely undocumented; some fonts from   */
        /* Arphic made for Chinese Windows 3.1 have this.                   */
        if(  face->format_tag != 0x00010000L &&    /* MS fonts  */
             face->format_tag != 0x00020000L &&    /* CJK fonts for Win 3.1 */
             face->format_tag != TTAG_true   &&    /* Mac fonts */
             face->format_tag != TTAG_OTTO   &&    /* OTTO */
             face->format_tag != TTAG_CFF )        /* CFF */
        {
            sfnt->done_face( face );
            itype_drv_destroy_mti_extension( ttface );
            ITC_PRINTF_DEBUG(( "[not a valid TTF font]\n" ));
            return ITYPE_DRV_Err_Unknown_File_Format;
        }

        /* If we are performing a simple font format check, exit immediately */
        if( face_index < 0 )
        {
            sfnt->done_face( face );
            itype_drv_destroy_mti_extension( ttface );
            return ITYPE_DRV_Err_Ok;
        }

        /* Load font directory */
        error = sfnt->load_face( stream, face, face_index, num_params, params );
        if( error )
        {
            sfnt->done_face( face );
            itype_drv_destroy_mti_extension( ttface );
            return error;
        }

#ifdef ITYPE_CCC_LTT_CFF_ONLY
        /* CFF (.cff, .otf) are OK, otherwise check if it is a CCC */
        if( face->format_tag != TTAG_OTTO )
        {
            FS_BOOLEAN cccFont = STIK_FORMAT_CCC(face->header.Glyph_Data_Format) ||
                                 TTF_FORMAT_CCC(face->header.Glyph_Data_Format) ||
                                 STIK_FORMAT_DDD(face->header.Glyph_Data_Format) ||
                                 TTF_FORMAT_DDD(face->header.Glyph_Data_Format);
            if( cccFont == FALSE )
            {
#ifdef ITYPE_EDGE_TTFS
                /* Edge TTF fonts are also OK */
                FT_ULong length = 0;

                error = sfnt->load_any( face, FT_MAKE_TAG('A','D','F','H'), 0, NULL, &length );
                if ( error != FT_Err_Ok )
                {
                    sfnt->done_face( face );
                    itype_drv_destroy_mti_extension( ttface );
                    return ITYPE_DRV_Err_Unknown_File_Format;
                }
#else
                sfnt->done_face( face );
                itype_drv_destroy_mti_extension( ttface );
                return ITYPE_DRV_Err_Unknown_File_Format;
#endif
            }
        }
#endif
    }
    else /* LTT font */
    {
        /* for a LTT font, we handle it differently - not using sfnt module */

#ifdef FS_LINKED_FONTS
        /*lint -fqb reverse the message: Note 963 */
        FsLtt *ltt;
        /*lint +fqb reverse the message: Note 963 */
        const FILECHAR *familyName, *subFamilyName;
        FT_ULong name_len;
        FsLttComponent *metricComponent;

        bIsLttFont = TRUE;
        face->format_tag = format_tag;
        ttface->num_faces = 1;

        /* we need to copy font name to ttface */
        ltt = FsLtt_new( FS_state_ptr );
        if( !ltt )
        {
            itype_drv_destroy_mti_extension( ttface );
            return ITYPE_DRV_Err_Unknown_File_Format;
        }

        fpLtt = idrv_fopen( sStream, "rb" );
        if ( !fpLtt )
        {
            FsLtt_delete( ltt );
            itype_drv_destroy_mti_extension( ttface );
            return ITYPE_DRV_Err_Unknown_File_Format;
        }
        itype_err = FsLtt_read( ltt, fpLtt, false );
        if (itype_err != SUCCESS)
        {
            idrv_fclose(fpLtt);
            itype_drv_destroy_mti_extension( ttface );
            return ITYPE_DRV_Err_Unknown_File_Format;
        }
        idrv_fclose(fpLtt);

        familyName = FsLtt_getName( ltt );
        name_len = strlen( familyName );
        if( FT_ALLOC( ttface->family_name, name_len + 1 ) )
        {
            FsLtt_delete( ltt );
            itype_drv_destroy_mti_extension( ttface );
            return ITYPE_DRV_Err_Out_Of_Memory;
        }
        strcat( ttface->family_name, familyName );

        subFamilyName = FsLtt_getSubfamilyName(ltt);
        name_len = strlen( subFamilyName );
        if( FT_ALLOC( ttface->style_name, name_len + 1 ) )
        {
            FT_FREE( ttface->family_name );
            FsLtt_delete( ltt );
            itype_drv_destroy_mti_extension( ttface );
            return ITYPE_DRV_Err_Out_Of_Memory;
        }
        strcat( ttface->style_name, subFamilyName );

        /* look for a component font that uses targetDir - set flag if any found */
        for (ix = 0; ix < FsLtt_numComponents(ltt); ix++)
        {
            if ( FsLtt_component(ltt, ix)->targetDir != NULL )
            {
                bIsLttDirFont = TRUE;
                break;
            }
        }

        metricComponent = FsLtt_component(ltt, ltt->mtrxComponent);
        ((MTI_Info)(ttface->extensions))->numGlyphMetricFont = metricComponent->numGlyphs;
        ((MTI_Info)(ttface->extensions))->lttMetricComponent = ltt->mtrxComponent;

        FsLtt_delete( ltt );
#else
#ifdef TXTC_BUILDS
    FS_BOOLEAN RTGAH_FLAG = true;
    FS_LONG flags = FS_get_flags( FS_state_ptr );

    if( (flags & FLAGS_AUTOHINT_ON) == 0 )
        RTGAH_FLAG = false;

    print_config_info( RTGAH_FLAG,
                       FS_state_ptr->server->heap_size,
                       FS_state_ptr->server->allocated );
    itype_drv_destroy_mti_extension( ttface );
    return ITYPE_DRV_Err_Unknown_File_Format;
#endif
#endif
        face->sfnt = sfnt;    /* storing sfnt in face which will be used only to find kern values in case of LTT font */
    }

    /* put together the font name and store in the extension */
    itype_drv_create_font_name( ttface );

    if( NULL == itype_drv_get_font_name( ttface ) )
    {
        if ( sfnt != NULL )
            sfnt->done_face( face );
        if( ttface->family_name )
            FT_FREE( ttface->family_name );
        if( ttface->style_name )
            FT_FREE( ttface->style_name );

        itype_drv_destroy_mti_extension( ttface );
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    /* ltt files that use targetDir are not supported with FS_Streams - if using one, pass the path - otherwise use the stream name */
    itype_err = FS_add_font( FS_state_ptr,
                             itype_drv_get_font_name( ttface ),
                             bIsLttDirFont ? stream->pathname.pointer : sStream, 
                             (FS_BYTE *)NULL, 
                             (FS_ULONG)face_index );

    if ( itype_err != SUCCESS )
    {
        if ( sfnt != NULL )
            sfnt->done_face( face );
        if( ttface->family_name )
            FT_FREE( ttface->family_name );
        if( ttface->style_name )
            FT_FREE( ttface->style_name );
        itype_drv_destroy_mti_extension( ttface );
        return ITYPE_DRV_Err_Out_Of_Memory;
    }

    itype_err = FS_set_font( FS_state_ptr, itype_drv_get_font_name( ttface ) );
    if( itype_err  != SUCCESS )
    {
        if( SUCCESS != FS_delete_font( FS_state_ptr, itype_drv_get_font_name( ttface ) ) )
            FS_delete_font( FS_state_ptr, ttface->family_name );
        if ( sfnt != NULL )
            sfnt->done_face( face );
        if( ttface->family_name )
            FT_FREE( ttface->family_name );
        if( ttface->style_name )
            FT_FREE( ttface->style_name );
        itype_drv_destroy_mti_extension( ttface );
        error = itype_drv_convert_itype_error( itype_err );
        return error;
    }

#ifdef TXTC_BUILDS
    /* write to logcat */
    if( !strcmp( itype_drv_get_font_name( ttface ), "givemeinfo givemeinfo" ) )
    {
        FS_BOOLEAN RTGAH_FLAG = true;

        FS_LONG flags = FS_get_flags( FS_state_ptr );
        if( (flags & FLAGS_AUTOHINT_ON) == 0 )
            RTGAH_FLAG = false;

        print_config_info( RTGAH_FLAG,
                           FS_state_ptr->server->heap_size,
                           FS_state_ptr->server->allocated );
    }
    /* write to logcat */
#endif

    if( bIsLttFont )
    {
        error = itype_drv_load_ltt_info( ttface, FS_state_ptr );
        if( error != ITYPE_DRV_Err_Ok )
        {
            itype_drv_destroy_mti_extension( ttface );
            return error;
        }
    }

    return ITYPE_DRV_Err_Ok;
}

/*************************************************************************/
/*                                                                       */
/*                          itype_drv_face_done()                        */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( void )
itype_drv_face_done( FT_Face ttface )
{
    TT_Face      face = (TT_Face)ttface;
    FT_Memory    memory = FT_FACE_MEMORY( face );
    SFNT_Service sfnt = (SFNT_Service)face->sfnt;

    if( VALID_MTI_EXTENSION(ttface->extensions) )
    {
        MTI_Info ext = (MTI_Info)ttface->extensions;

        if ( ext->outline != NULL )
        {
            FT_FREE( ext->outline->points );
            FT_FREE( ext->outline->tags );
            FT_FREE( ext->outline->contours );
            FT_FREE( ext->outline );
            ext->outline = NULL;
        }

        if(face->format_tag == 0x4c54540a || face->format_tag == 0x4c54540d) /* freeing kern table and setting sfnt to null in case of LTT fonts */
        {
            FS_free_table(ext->face_client, face->kern_table);
            sfnt = NULL;    /* setting sfnt to NULL so that done_face is not called for LTT */
            FT_FREE( face->dir_tables );
            face->num_tables = 0;
        }

        /* delete the font */
        if( ext->face_client != NULL )
        {
            /* for true type font, use full name to delete the font, but linked font use family name  */
            /* if can't delete the font, try family name: ltt case! 601-font in use, 301-font not found  */
            if( SUCCESS != FS_delete_font( ext->face_client, itype_drv_get_font_name( ttface ) ) )
                FS_delete_font( ext->face_client, ttface->family_name );
        }

        if( ttface->family_name )
            FT_FREE( ttface->family_name );
        if( ttface->style_name )
            FT_FREE( ttface->style_name );

        if( sfnt )
            sfnt->done_face( face );

        /* destroy the extension (and the client iType state) */
        itype_drv_destroy_mti_extension( ttface );
    }
}

#ifdef FT_CONFIG_OPTION_OLD_INTERNALS
/*************************************************************************/
/*                                                                       */
/*                         itype_drv_set_point_size()                    */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_set_point_size( FT_Size size,
                          FT_F26Dot6 char_width,
                          FT_F26Dot6 char_height,
                          FT_UInt horz_resolution,
                          FT_UInt vert_resolution )
{
    FT_Error error;
    FS_LONG itype_err;

    FT_UNUSED(vert_resolution);
    FT_UNUSED(horz_resolution);
    FT_UNUSED(char_height);
    FT_UNUSED(char_width);
    itype_err = itype_drv_set_itype_size( size, FT_LOAD_DEFAULT );
    error = itype_drv_convert_itype_error( itype_err );
    return error;
}

/*************************************************************************/
/*                                                                       */
/*                        itype_drv_set_pixel_size()                     */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_set_pixel_size( FT_Size size,
                          FT_UInt pixel_width,
                          FT_UInt pixel_height )
{
    FT_Error error;
    FS_LONG itype_err;

    FT_UNUSED(pixel_height);
    FT_UNUSED(pixel_width);

    itype_err = itype_drv_set_itype_size( size, FT_LOAD_DEFAULT );
    error = itype_drv_convert_itype_error( itype_err );

    return error;
}
#endif

/*************************************************************************/
/*                                                                       */
/*                          itype_drv_load_glyph()                       */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_load_glyph( FT_GlyphSlot slot,
                      FT_Size      size,
                      FT_UInt      glyph_index,
                      FT_Int32     load_flags )
{
    FT_Error  error = ITYPE_DRV_Err_Ok;
    FS_LONG   itype_err = SUCCESS;
    FS_STATE  *FS_state_ptr;
    FT_Face   face;
    MTI_Info ext;

    face = slot->face;

    if( !VALID_MTI_EXTENSION(slot->face->extensions) )
        return ITYPE_DRV_Err_Unknown_File_Format;

    ext = (MTI_Info)slot->face->extensions;
    FS_state_ptr = ext->face_client;

    if( FS_state_ptr == NULL )
        return ITYPE_DRV_Err_Unknown_File_Format;

    /* resolve load_flags conflicts, see freetype related code in ttdriver */
    if( load_flags & FT_LOAD_NO_HINTING )
    {
      /* both FT_LOAD_NO_HINTING and FT_LOAD_NO_AUTOHINT   */
      /* are necessary to disable hinting for tricky fonts */

      if( FT_IS_TRICKY( face ) )
          load_flags &= ~FT_LOAD_NO_HINTING;

      if( load_flags & FT_LOAD_NO_AUTOHINT )
          load_flags |= FT_LOAD_NO_HINTING;
    }

    if( load_flags & ( FT_LOAD_NO_RECURSE | FT_LOAD_NO_SCALE ) )
    {
        load_flags |= FT_LOAD_NO_BITMAP | FT_LOAD_NO_SCALE;

        if( !FT_IS_TRICKY( face ) )
            load_flags |= FT_LOAD_NO_HINTING;
    }

    /* resolve load_flags conflicts, see freetype related code in ttdriver */

    /* we don't need to call FS_set_cmap(), because we have glyph index here */
    FS_set_bold_pct( FS_state_ptr, 0 );
    FS_set_stroke_pct( FS_state_ptr, DEFAULT_STROKE_PCT );
    FS_set_flags( FS_state_ptr, FLAGS_CMAP_OFF );

    if( load_flags & FT_LOAD_NO_HINTING )
    {
        FS_set_flags( FS_state_ptr, FLAGS_HINTS_OFF);
        FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_OFF );
        FS_set_flags( FS_state_ptr, FLAGS_FORCE_AUTOHINT_OFF );
        FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_YONLY_OFF );
    }
    else /* hinting on */
    {
        FS_set_flags( FS_state_ptr, FLAGS_HINTS_ON );
        if( load_flags & FT_LOAD_FORCE_AUTOHINT )      /* force autohinting on */
        {
            FS_set_flags( FS_state_ptr, FLAGS_GRAYSCALE );
            FS_set_flags( FS_state_ptr, FLAGS_FORCE_AUTOHINT );
            FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_YONLY_OFF );
        }
        else /* force autohinting off */
        {
#ifndef FS_AUTOHINT_DEFAULT_ON
            if( (load_flags & FT_LOAD_NO_AUTOHINT) ) /* autohinting off */
            {
                FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_OFF );
                FS_set_flags( FS_state_ptr, FLAGS_FORCE_AUTOHINT_OFF );
                FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_YONLY_OFF );
            }
            else /* autohinting on */
#endif
            {
                FT_Render_Mode  mode;
                FS_set_flags( FS_state_ptr, FLAGS_GRAYSCALE );
                mode = FT_LOAD_TARGET_MODE( load_flags );
                if( mode == FT_RENDER_MODE_LIGHT )
                {
                    FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_ON );
                    FS_set_flags( FS_state_ptr, FLAGS_FORCE_AUTOHINT );
                    FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_YONLY_ON );
                }
                else
                {
                    FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_ON );
                    FS_set_flags( FS_state_ptr, FLAGS_FORCE_AUTOHINT_OFF );
                    FS_set_flags( FS_state_ptr, FLAGS_AUTOHINT_YONLY_OFF );
                }
            }
        }
    }

    /* check to see if we need to call set_scale() or not */
    if( (FS_state_ptr->cur_sfnt) == NULL ||
        (size->metrics.x_ppem != FS_ROUND(FS_state_ptr->scale[0])) ||
        (size->metrics.y_ppem != FS_ROUND(FS_state_ptr->scale[3])) )
    {
        itype_err = itype_drv_set_itype_size( size, load_flags );
        if( itype_err != SUCCESS )
        {
            ITC_PRINTF_DEBUG(( "\niType error occurred %ld", itype_err ));
            error = itype_drv_convert_itype_error( itype_err );
        }
    }


    {
        /*  Get the outline using the current load flags. */

        FS_OUTLINE *outl;

        /* set the load flags, they are used in the renderer */
        ext->load_flags = load_flags;

        /* There should be only one vertical writing flag set at a time.
           In case there is more than one, the priority order is FT_LOAD_VERTICAL_LAYOUT,
           FT_LOAD_VERTICAL_ROTATE_RIGHT, FT_LOAD_VERTICAL_ROTATE_LEFT */
        if ( (load_flags & FT_LOAD_VERTICAL_LAYOUT) != 0 )
            FS_set_flags( FS_state_ptr, FLAGS_VERTICAL_ON );
        else if ( (load_flags & FT_LOAD_VERTICAL_ROTATE_RIGHT) != 0 )
            FS_set_flags( FS_state_ptr, FLAGS_VERTICAL_ROTATE_RIGHT_ON );
        else if ( (load_flags & FT_LOAD_VERTICAL_ROTATE_LEFT) != 0 )
            FS_set_flags( FS_state_ptr, FLAGS_VERTICAL_ROTATE_LEFT_ON );
        else
            FS_set_flags( FS_state_ptr, FLAGS_VERTICAL_OFF ); /* turns all vertical writing off */


        /* get the outline */
        outl = FS_get_outline( FS_state_ptr, glyph_index );
        if( outl == NULL )
        {
            error = ITYPE_DRV_Err_Invalid_Argument;
        }
        else
        {
#ifdef FS_STIK
            if ((outl->num > 0) &&
                (STATE.cur_lfnt->fontflags & FONTFLAG_STIK) &&
                !(outl->type[0] & OUTLINE_CHAR))
            {
                FS_LONG npoints, ntypes;
                FS_OUTLINE *expo;
                expo = idrv_expand_stik( FS_state_ptr, outl, &ntypes, &npoints );

                FSS_free_char( FS_state_ptr, outl );
                if ( expo == NULL )
                    error = ITYPE_DRV_Err_iType_Memory_Malloc_Error;

                outl = expo;
            }
#endif

#ifdef TXTC_BUILDS
            if (strncmp( FS_state_ptr->cur_typeset.fntset->name,"givemeinfo",20)==0 ) /* test font */
            {
                FILECHAR    *lfnt_name = FS_state_ptr->cur_lfnt->name;

                if ( !strcmp( lfnt_name, "Univers OTS" ) ) /* 1st component */
#ifdef FS_HINTS
                    print_RTGAH_info( FS_state_ptr->error, FS_state_ptr->any_hints );
#endif
                if ( !strcmp( lfnt_name, "MHeiGB18030C-Medium_E" ) ) /* 2nd component */
                {
                    __android_log_print(ANDROID_LOG_INFO, "ITYPE_STATEERROR2", "%ld", FS_state_ptr->error);
#ifdef FS_EDGE_HINTS
                    __android_log_print(ANDROID_LOG_INFO, "ITYPE_EDGEHINTING", "%d", FS_state_ptr->any_hints);
#endif
                }
            }
#endif

            if ( error == ITYPE_DRV_Err_Ok )
            {
                itype_drv_fill_in_slot_outline( slot, outl, load_flags );
                itype_drv_add_raster_params( slot, FS_state_ptr, glyph_index );
                itype_drv_fill_in_slot_metrics( slot, FS_state_ptr, outl, load_flags, (FS_USHORT)glyph_index );

                error = FT_Outline_Check( &slot->outline );
                if( error )
                    error = ITYPE_DRV_Err_Invalid_Argument;
            }
        } /* outlh not NULL */

        FS_free_char( FS_state_ptr, outl );
    }

    if( itype_err == SUCCESS )
        FS_set_flags( FS_state_ptr, FLAGS_CMAP_ON );

    return error;
}

/*************************************************************************/
/*                                                                       */
/*                          itype_drv_get_kerning()                      */
/*                                                                       */
/*************************************************************************/
FT_LOCAL_DEF( FT_Error )
itype_drv_get_kerning( FT_Face    ttface,
                       FT_UInt    left_glyph,
                       FT_UInt    right_glyph,
                       FT_Vector *kerning )
{
    FT_Error error = ITYPE_DRV_Err_Ok;
    SFNT_Service  sfnt;

    TT_Face face = (TT_Face) ttface;

    kerning->x = 0;
    kerning->y = 0;

    if(face->format_tag == 0x4c54540a || face->format_tag == 0x4c54540d)    /* check for LTT Font */
    {
        MTI_Info ext = (MTI_Info)ttface->extensions;
        if((left_glyph >= ext->numGlyphMetricFont && right_glyph >= ext->numGlyphMetricFont) || ext->lttMetricComponent != 0)
        {
                /* Call FS_get_kerning if both left and right glyph does not belong to metric font */
                FS_STATE *FS_state_ptr;
                FS_LONG itype_err;
                FS_FIXED kernX, kernY;

                FS_state_ptr = ext->face_client;
                itype_err = FS_set_flags(FS_state_ptr, FLAGS_CMAP_OFF);
                if( itype_err == SUCCESS )
                {
                    itype_err = FS_set_flags(FS_state_ptr, FLAGS_KERNING_SCALE_OFF);
                    if( itype_err == SUCCESS )
                    {
                        itype_err = FS_set_scale(FS_state_ptr, 1<<16, 0, 0, 1<<16);
                        if( itype_err == SUCCESS )
                        {
                            itype_err = FS_get_kerning(FS_state_ptr, left_glyph, right_glyph, &kernX, &kernY);
                            if( itype_err == SUCCESS )
                            {
                                kerning->x = (FT_Pos)kernX;
                                itype_err = FS_set_flags(FS_state_ptr, FLAGS_CMAP_ON );
                                if( itype_err == SUCCESS )
                                {
                                    itype_err = FS_set_flags(FS_state_ptr, FLAGS_KERNING_SCALE_ON);
                                }
                            }
                        }
                    }
                    if( itype_err != SUCCESS )
                    {
                        ITC_PRINTF_DEBUG(( "\niType error occurred %ld", itype_err ));
                        error = itype_drv_convert_itype_error(itype_err);
                    }
                }
                return error;
        }
        else if((left_glyph >= ext->numGlyphMetricFont) || (right_glyph >= ext->numGlyphMetricFont))    /* return if either left or right glyph does not belong to the metric font */
        {
            return ITYPE_DRV_Err_Ok;
        }
    }

    sfnt = face->sfnt;
    if ( sfnt )
        kerning->x = sfnt->get_kerning( face, left_glyph, right_glyph );

    return ITYPE_DRV_Err_Ok;
}

/*************************************************************************/
/*                                                                       */
/*                       itype driver module interface                   */
/*                                                                       */
/*************************************************************************/

FT_Error
itype_drv_set_csm_adjustments( FT_Face face,
                               FT_Fixed sharpness_offset,
                               FT_Fixed sharpness_slope,
                               FT_Fixed thickness_offset,
                               FT_Fixed thickness_slope)
{
    MTI_Info ext;

    if( face == NULL )
        return FT_Err_Invalid_Argument;

    if( !VALID_MTI_EXTENSION(face->extensions) )
        return FT_Err_Invalid_Argument;

    ext = (MTI_Info)face->extensions;
    ext->sharpness_offset = sharpness_offset;
    ext->sharpness_slope = sharpness_slope;
    ext->thickness_offset = thickness_offset;
    ext->thickness_slope = thickness_slope;

    return FT_Err_Ok;
}

FT_Error itype_drv_get_csm_adjustments( FT_Face face,
                                        FT_Fixed *sharpness_offset,
                                        FT_Fixed *sharpness_slope,
                                        FT_Fixed *thickness_offset,
                                        FT_Fixed *thickness_slope )
{
    MTI_Info ext;

    if( face == NULL )
        return FT_Err_Invalid_Argument;
    if( !sharpness_offset ||
         !sharpness_slope  ||
         !thickness_offset ||
         !thickness_slope )
        return FT_Err_Invalid_Argument;

    if( !VALID_MTI_EXTENSION(face->extensions) )
        return FT_Err_Invalid_Argument;

    ext = (MTI_Info)face->extensions;

    *sharpness_offset = ext->sharpness_offset;
    *sharpness_slope = ext->sharpness_slope;
    *thickness_offset = ext->thickness_offset;
    *thickness_slope = ext->thickness_slope;

    return FT_Err_Ok;
}

const itype_module_interfaceRec idrv_interface =
{
    itype_drv_set_csm_adjustments,
    itype_drv_get_csm_adjustments
};


/*************************************************************************/
/*                                                                       */
/*                       itype_driver_class                              */
/*                                                                       */
/*  This is the itype font driver module which is based on the           */
/*  FT_DriverInterface structure.                                        */
/*************************************************************************/
FT_CALLBACK_TABLE_DEF
/*lint +fqb to reverse the message. Note 963: Qualifier const or volatile precedes a type; */
const FT_Driver_ClassRec  itype_driver_class =
{
    {
      FT_MODULE_FONT_DRIVER |
   /* FT_MODULE_DRIVER_SCALABLE |*//* undef idrv as scalable font driver to avoid 
                                      FT_Load_Glyph() to apply freetype auto-hinting */
      FT_MODULE_DRIVER_HAS_HINTER, /* bitflags describing the module */

      sizeof ( FT_DriverRec ),     /* the size of one module in bytes */

      ITYPE_DRIVER_NAME,           /* driver module name */
      ITYPE_DRIVER_VERSION,        /* driver module version */
      0x20000L,                    /* driver module requires FreeType 2.0 or above */

      &idrv_interface,             /* driver module specific interface */

      itype_drv_module_init,       /* function to initialize not create the module */
      itype_drv_module_done,       /* function used to finalize not destroy the module */
      itype_drv_module_get_interface, /* queries the module for a specific interface by name */
    },

    sizeof ( TT_FaceRec ),         /* Face object size */
    sizeof ( FT_SizeRec ),         /* Size object size */
    sizeof ( FT_GlyphSlotRec ),    /* Slot object size */

    itype_drv_face_init,           /* format specific face constructor */
    itype_drv_face_done,           /* format specific face destructor */
    0,                             /* format specific size constructor */
    0,                             /* format specific size destructor */
    0,                             /* format specific slot constructor */
    0,                             /* format specific slot destructor */

#ifdef FT_CONFIG_OPTION_OLD_INTERNALS
    itype_drv_set_point_size,      /* handle to a function used to set the new */
                                   /* character size in points + resolution */
    itype_drv_set_pixel_size,      /* handle to a function used to set the new */
                                   /* character size in pixels */
#endif
    itype_drv_load_glyph,          /* handle to a function used to load a glyph */
                                   /* image into the slot */
    itype_drv_get_kerning,         /* handle to a function used to return the */
                                   /* unscaled kerning for a given pair of glyphs */
    0,                             /* FT_Face_AttachFunc */
    0,                             /* tt_get_advances */
    0,                             /* tt_size_request */
#ifdef TT_CONFIG_OPTION_EMBEDDED_BITMAPS
    0,                             /* tt_size_select */
#else
    0                              /* FT_Size_SelectFunc */
#endif
};

/*************************************************************************/
/*                                                                       */
/*                   itype_drv_dump_itype_outline()                      */
/*                                                                       */
/*************************************************************************/
#ifdef DUMP_OUTLINES
FS_VOID
itype_drv_dump_itype_outline( FS_OUTLINE *outl, char *msg )
{
    int i;
    FS_FIXED *x,*y;

    if (!outl)
        return;

    printf( "\n" );
    printf("----------------------------------------------------------\n");
    printf( "iType Drive dump FS_OUTLINE: %s\n", msg );
    printf("----------------------------------------------------------\n");

    /* header */
    printf( "num_types = %d, num_contours = %d\n", outl->num, outl->nc );
    printf("xmin=%5.5f  xmax=%5.5f  ymin=%5.5f  ymax=%5.5f\n",
              outl->lo_x/65536.0, outl->hi_x/65536.0, 
              outl->lo_y/65536.0, outl->hi_y/65536.0);

    printf("i_dx=%d  i_dy=%d  dx=%5.5f  dy=%5.5f\n",
              outl->i_dx, outl->i_dy, outl->dx/65536.0, outl->dy/65536.0);
    printf("----------------------------------------------------------\n");

    /* now the curve data */
    x = outl->x;
    y = outl->y;

    for (i=0; i<outl->num; i++)
        {
        switch( outl->type[i] & 0x7f)
            {
            case FS_MOVETO:
                printf("%d %12.5f %12.5f moveto\n",i,*x++ / 65536.0, *y++ / 65536.0);
                break;

            case FS_LINETO:
                printf("%d %12.5f %12.5f lineto\n",i,*x++ / 65536.0, *y++ / 65536.0);
                break;

            case FS_QUADTO:
                printf("%d %12.5f %12.5f\n",i,*x++ / 65536.0, *y++ / 65536.0);
                printf("%d %12.5f %12.5f quadto\n",i,*x++ / 65536.0, *y++ / 65536.0);
                break;

            case FS_CUBETO:
                printf("%d %12.5f %12.5f\n",i,*x++ / 65536.0, *y++ / 65536.0);
                printf("%d %12.5f %12.5f\n",i,*x++ / 65536.0, *y++ / 65536.0);
                printf("%d %12.5f %12.5f cubeto\n",i,*x++ / 65536.0, *y++ / 65536.0);
                break;
            }
        }
    printf("\n");
    fflush(stdout);
}
#endif


/*************************************************************************/
/*                                                                       */
/*                   itype_drv_dump_freetype_outline()                   */
/*                                                                       */
/*************************************************************************/
#ifdef DUMP_OUTLINES
void
itype_drv_dump_freetype_outline( FT_Outline *outl, char *msg )
{
    int i = 0;

    printf( "\n" );
    printf("----------------------------------------------------------\n");
    printf( "iType Drive dump FT_Outline: %s\n", msg );
    printf("----------------------------------------------------------\n");
    printf( "num_points = %d, num_contours = %d\n", outl->n_points, outl->n_contours );
    printf( "index          x          y          type\n");
    printf("----------------------------------------------------------\n");

    for( i = 0; i < outl->n_points; i++ )
    {
        printf("  %d     %12.5f     %12.5f", i, outl->points[i].x/64.0, outl->points[i].y/64.0);
        switch(FT_CURVE_TAG(outl->tags[i]))
        {
        case FT_CURVE_TAG_ON:
            printf("  %s\n","FT_CURVE_TAG_ON");
            break;
        case FT_CURVE_TAG_CONIC:
            printf("  %s\n","FT_CURVE_TAG_CONIC");
            break;
        case FT_CURVE_TAG_CUBIC:
            printf("  %s\n","FT_CURVE_TAG_CUBIC");
            break;
        default:
            printf("  %s\n","Unknown");
            break;
        }
    }

    for(i = 0; i < outl->n_contours; i++)
    {
        printf("contour %d  index %d  x %12.5f    y %12.5f\n",i ,
                              outl->contours[i], 
                              outl->points[outl->contours[i]].x/64.0,
                              outl->points[outl->contours[i]].y/64.0);
    }
}
#endif

#ifdef TXTC_BUILDS
/* write to logcat */
void print_config_info( FS_BOOLEAN RTGAH_FLAG, FS_ULONG heap_size, FS_ULONG allocated )
{
    int configuration = 0;

#ifdef FS_MULTI_PROCESS
    configuration |= 0x00000001;
#endif
#ifdef FS_MULTI_THREAD
    configuration |= 0x00000002;
#endif
#ifdef FS_INT_MEM
    configuration |= 0x00000004;
#endif
#ifdef FS_EMBEDDED_BITMAP
    configuration |= 0x00000008;
#endif
#ifdef FS_BITMAPS
    configuration |= 0x00000010;
#endif
#ifdef FS_GRAYMAPS
    configuration |= 0x00000020;
#endif
#ifdef FS_EDGE_HINTS
    configuration |= 0x00000040;
#endif
#ifdef FS_EDGE_RENDER
    configuration |= 0x00000080;
#endif
#ifdef FS_HINTS
    configuration |= 0x00000100;
#endif
#ifdef FS_LINKED_FONTS
    configuration |= 0x00000200;
#endif
#ifdef FS_STIK
    configuration |= 0x00000400;
#endif
#ifdef FS_CCC
    configuration |= 0x00000800;
#endif
#ifdef FS_ACT3
    configuration |= 0x00001000;
#endif
#ifdef FS_PHASED
    configuration |= 0x00002000;
#endif
#ifdef FS_ICONS
    configuration |= 0x00004000;
#endif
#ifdef FS_PSEUDO_BOLD
    configuration |= 0x00008000;
#endif
#ifdef FS_CACHE_ADVANCE
    configuration |= 0x00010000;
#endif
#ifdef FS_CACHE_OUTLINES
    configuration |= 0x00020000;
#endif
#ifdef FS_CACHE_BITMAPS
    configuration |= 0x00040000;
#endif
#ifdef FS_CACHE_GRAYMAPS
    configuration |= 0x00080000;
#endif
#ifdef FS_CACHE_PHASED
    configuration |= 0x00100000;
#endif
#ifdef FS_CACHE_EDGE_GLYPHS
    configuration |= 0x00200000;
#endif
#ifdef FS_CACHE_CMAP
    configuration |= 0x00400000;
#endif
#ifdef  FS_CONTOUR_WINDING_DETECTION
    configuration |= 0x00800000;
#endif
#ifdef  FS_STIK_EVENS_ONLY
    configuration |= 0x01000000;
#endif
#ifdef  FS_OPENVG
    configuration |= 0x02000000;
#endif
#ifdef  FS_PFRR
    configuration |= 0x04000000;
#endif
#ifdef  FS_DEBUG
    configuration |= 0x08000000;
#endif
#ifdef  FS_MP_DEBUG
    configuration |= 0x10000000;
#endif
#ifdef  FS_MEM_DEBUG
    configuration |= 0x20000000;
#endif
#ifdef  FS_DUMP
    configuration |= 0x40000000;
#endif
#ifdef  FS_STATS
    configuration |= 0x80000000;
#endif

    __android_log_print(ANDROID_LOG_INFO, "ITYPE_CONFIGURATION", "%x", configuration);

    __android_log_print(ANDROID_LOG_INFO, "ITYPE_HEAPSIZE", "%ld", heap_size);

    __android_log_print(ANDROID_LOG_INFO, "ITYPE_ALLOCATED", "%ld", allocated);


    if( RTGAH_FLAG == true )
        __android_log_print(ANDROID_LOG_INFO, "RTGAH_USED", "%s", "0x00000008");
}

/* write to logcat for RTGAH */
#ifdef FS_HINTS
void print_RTGAH_info( FS_ULONG error, FS_BYTE hints )
{
    __android_log_print(ANDROID_LOG_INFO, "ITYPE_STATEERROR1", "%ld", error);
    __android_log_print(ANDROID_LOG_INFO, "ITYPE_RTGAHHINTING", "%d", hints);
}
#endif
#endif
